<?php
/**
 * Plugin Name:       Flexi Recent Posts
 * Plugin URI:        https://mwahyunz.id/plugin/flexi-recent-posts/
 * Description:       Display a list of recent posts flexibly anywhere using the [flexiposts] shortcode with customizable attributes.
 * Version:           2.4.0
 * Requires at least: 6.8
 * Requires PHP:      8.0
 * Author:            Mhd Wahyu NZ
 * Author URI:        https://mwahyunz.id
 * License:           GPL v2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       frp
 * Domain Path:       /lang
 */

// Mencegah akses langsung ke file
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Define plugin constants
define( 'FRP_VERSION', '2.4.0' );
define( 'FRP_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'FRP_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

// Memuat file-file yang dibutuhkan
require_once FRP_PLUGIN_DIR . 'frp-display.php';

/**
 * Memuat stylesheet plugin.
 */
function frp_enqueue_styles() {
    wp_enqueue_style(
        'frp-style',
        FRP_PLUGIN_URL . 'frp-style.css',
        array(),
        FRP_VERSION
    );
}
add_action( 'wp_enqueue_scripts', 'frp_enqueue_styles' );

/**
 * Memuat text domain untuk terjemahan.
 */
function frp_load_textdomain() {
    load_plugin_textdomain( 'frp', false, dirname( plugin_basename( __FILE__ ) ) . '/lang/' );
}
add_action( 'plugins_loaded', 'frp_load_textdomain' );

/**
 * Validasi dan sanitasi atribut shortcode
 *
 * @param array $atts Atribut dari shortcode
 * @return array Atribut yang telah divalidasi
 */
function frp_validate_shortcode_atts( $atts ) {
    $validated = array();
    
    // Boolean attributes (0 or 1)
    $boolean_atts = array( 'title', 'date', 'cat', 'tag', 'excerpt', 'author', 'image', 'authorlink', 'more' );
    foreach ( $boolean_atts as $att ) {
        $validated[ $att ] = isset( $atts[ $att ] ) ? (int) (bool) $atts[ $att ] : 1;
    }
    
    // Atribut 'more' default 0
    $validated['more'] = isset( $atts['more'] ) ? (int) (bool) $atts['more'] : 0;
    
    // Number attributes dengan validasi range
    $validated['number'] = isset( $atts['number'] ) ? max( 1, min( 100, absint( $atts['number'] ) ) ) : 3;
    $validated['limit'] = isset( $atts['limit'] ) ? max( 1, min( 1000, absint( $atts['limit'] ) ) ) : 80;
    $validated['imagesize'] = isset( $atts['imagesize'] ) ? max( 0, min( 2000, absint( $atts['imagesize'] ) ) ) : 100;
    
    // Mode attribute (1, 2, or 3)
    $validated['mode'] = isset( $atts['mode'] ) ? max( 1, min( 3, absint( $atts['mode'] ) ) ) : 1;
    
    return $validated;
}

/**
 * Fungsi utama untuk shortcode [flexiposts].
 *
 * @param array $atts Atribut yang diberikan oleh pengguna dalam shortcode.
 * @return string HTML output dari daftar tulisan.
 */
function frp_shortcode_handler( $atts ) {
    // Menetapkan nilai default untuk atribut
    $default_atts = array(
        'title'      => 1,
        'date'       => 1,
        'cat'        => 1,
        'tag'        => 1,
        'excerpt'    => 1,
        'author'     => 1,
        'image'      => 1,
        'number'     => 3,
        'limit'      => 80,
        'mode'       => 1,
        'authorlink' => 1,
        'imagesize'  => 100,
        'more'       => 0,
    );
    
    $atts = shortcode_atts( $default_atts, $atts, 'flexiposts' );
    
    // Validasi dan sanitasi
    $atts = frp_validate_shortcode_atts( $atts );

    // Membuat kunci cache yang unik berdasarkan atribut
    $transient_key = 'frp_posts_' . md5( serialize( $atts ) . FRP_VERSION );

    // Coba ambil hasil dari cache (transient)
    $cached_output = get_transient( $transient_key );
    if ( false !== $cached_output && is_string( $cached_output ) ) {
        return $cached_output;
    }

    // Argumen untuk WP_Query dengan keamanan tambahan
    $query_args = array(
        'post_type'           => 'post',
        'posts_per_page'      => $atts['number'],
        'post_status'         => 'publish',
        'ignore_sticky_posts' => true,
        'no_found_rows'       => true, // Performance: tidak perlu pagination
        'update_post_meta_cache' => false, // Performance: tidak perlu meta cache
        'update_post_term_cache' => true, // Tetap perlu untuk kategori/tag
    );

    $recent_posts_query = new WP_Query( $query_args );

    // Memulai output buffering untuk menangkap HTML
    ob_start();

    if ( $recent_posts_query->have_posts() ) {
        echo '<div class="frp-container">';

        while ( $recent_posts_query->have_posts() ) {
            $recent_posts_query->the_post();
            // Memanggil fungsi display dari file frp-display.php
            frp_render_post_display( get_post(), $atts );
        }

        echo '</div>';
    } else {
        // Pesan jika tidak ada tulisan yang ditemukan
        echo '<p>' . esc_html__( 'No recent posts found.', 'frp' ) . '</p>';
    }

    // Mengembalikan data post global ke kondisi semula
    wp_reset_postdata();

    // Mengambil konten dari buffer
    $output = ob_get_clean();

    // Validasi output sebelum caching
    if ( is_string( $output ) && ! empty( $output ) ) {
        // Simpan output ke dalam cache selama 1 jam
        set_transient( $transient_key, $output, HOUR_IN_SECONDS );
    }

    return $output;
}
add_shortcode( 'flexiposts', 'frp_shortcode_handler' );

/**
 * Fungsi untuk menghapus transient saat post disimpan/diperbarui/dihapus
 * agar daftar tulisan terbaru selalu update.
 */
function frp_flush_transients() {
    global $wpdb;
    
    // Menggunakan prepared statement untuk keamanan
    $pattern = $wpdb->esc_like( '_transient_frp_posts_' ) . '%';
    
    $transients = $wpdb->get_col(
        $wpdb->prepare(
            "SELECT option_name FROM {$wpdb->options} WHERE option_name LIKE %s",
            $pattern
        )
    );
    
    // Hapus setiap transient secara individual untuk kompatibilitas dengan object cache
    if ( is_array( $transients ) && ! empty( $transients ) ) {
        foreach ( $transients as $transient ) {
            $key = str_replace( '_transient_', '', $transient );
            delete_transient( $key );
        }
    }
}
add_action( 'save_post_post', 'frp_flush_transients' );
add_action( 'deleted_post', 'frp_flush_transients' );
add_action( 'trashed_post', 'frp_flush_transients' );
add_action( 'untrashed_post', 'frp_flush_transients' );

/**
 * Cleanup saat plugin dinonaktifkan
 */
function frp_deactivation_cleanup() {
    frp_flush_transients();
}
register_deactivation_hook( __FILE__, 'frp_deactivation_cleanup' );