<?php
// Mencegah akses langsung ke file
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Menghasilkan HTML untuk daftar term (kategori atau tag) dengan tautan yang membuka tab baru.
 *
 * @param int    $post_id ID tulisan.
 * @param string $taxonomy Nama taksonomi ('category' atau 'post_tag').
 * @return string HTML dari daftar term.
 */
function frp_get_the_term_list_new_tab( $post_id, $taxonomy ) {
    // Validasi input
    $post_id = absint( $post_id );
    if ( ! $post_id || ! taxonomy_exists( $taxonomy ) ) {
        return '';
    }
    
    $terms = get_the_terms( $post_id, $taxonomy );
    
    if ( is_wp_error( $terms ) || empty( $terms ) || ! is_array( $terms ) ) {
        return '';
    }
    
    $term_links = array();
    
    foreach ( $terms as $term ) {
        if ( ! is_object( $term ) || ! isset( $term->term_id ) ) {
            continue;
        }
        
        $link = get_term_link( $term, $taxonomy );
        
        if ( is_wp_error( $link ) ) {
            continue;
        }
        
        $term_links[] = sprintf(
            '<a href="%s" target="_blank" rel="noopener noreferrer">%s</a>',
            esc_url( $link ),
            esc_html( $term->name )
        );
    }
    
    return ! empty( $term_links ) ? implode( ', ', $term_links ) : '';
}

/**
 * Merender tampilan HTML untuk satu tulisan.
 *
 * @param WP_Post $post Objek post WordPress.
 * @param array   $atts Atribut dari shortcode.
 */
function frp_render_post_display( $post, $atts ) {
    // Validasi objek post
    if ( ! $post instanceof WP_Post ) {
        return;
    }
    
    // Ekstrak dan validasi atribut
    $show_title = ! empty( $atts['title'] );
    $show_date = ! empty( $atts['date'] );
    $show_cat = ! empty( $atts['cat'] );
    $show_tag = ! empty( $atts['tag'] );
    $show_excerpt = ! empty( $atts['excerpt'] );
    $show_author = ! empty( $atts['author'] );
    $show_image = ! empty( $atts['image'] );
    $limit_excerpt = absint( $atts['limit'] );
    $mode = absint( $atts['mode'] );
    $author_is_linked = ! empty( $atts['authorlink'] );
    $image_size_px = absint( $atts['imagesize'] );
    $show_more = ! empty( $atts['more'] );

    $post_id = absint( $post->ID );
    if ( ! $post_id ) {
        return;
    }
    
    $post_link = get_permalink( $post_id );
    
    if ( ! $post_link ) {
        return;
    }

    // --- Menyiapkan elemen-elemen ---

    // Judul
    $title_html = '';
    if ( $show_title ) {
        $title = get_the_title( $post_id );
        if ( $title ) {
            $title_html = sprintf(
                '<h3 class="frp-title"><a href="%s">%s</a></h3>',
                esc_url( $post_link ),
                esc_html( $title )
            );
        }
    }

    // Kategori & Tag
    $categories_html = $show_cat ? frp_get_the_term_list_new_tab( $post_id, 'category' ) : '';
    $tags_html = $show_tag ? frp_get_the_term_list_new_tab( $post_id, 'post_tag' ) : '';
    $terms_html = '';
    
    if ( $categories_html && $tags_html ) {
        $terms_html = sprintf(
            '<div class="frp-terms"><span class="frp-categories">%s</span><span class="frp-separator"> | </span><span class="frp-tags">%s</span></div>',
            $categories_html,
            $tags_html
        );
    } elseif ( $categories_html ) {
        $terms_html = sprintf(
            '<div class="frp-terms"><span class="frp-categories">%s</span></div>',
            $categories_html
        );
    } elseif ( $tags_html ) {
        $terms_html = sprintf(
            '<div class="frp-terms"><span class="frp-tags">%s</span></div>',
            $tags_html
        );
    }

    // Penulis & Tanggal
    $author_html = '';
    if ( $show_author ) {
        $author_id = absint( $post->post_author );
        $author_name = get_the_author_meta( 'display_name', $author_id );
        
        if ( $author_name ) {
            if ( $author_is_linked ) {
                $author_link = get_author_posts_url( $author_id );
                if ( $author_link ) {
                    $author_html = sprintf(
                        '<span class="frp-author"><a href="%s" target="_blank" rel="noopener noreferrer">%s</a></span>',
                        esc_url( $author_link ),
                        esc_html( $author_name )
                    );
                }
            } else {
                $author_html = sprintf(
                    '<span class="frp-author">%s</span>',
                    esc_html( $author_name )
                );
            }
        }
    }
    
    $date_html = '';
    if ( $show_date ) {
        $date = get_the_date( '', $post_id );
        if ( $date ) {
            $date_html = sprintf(
                '<span class="frp-date">%s</span>',
                esc_html( $date )
            );
        }
    }
    
    $meta_html = '';
    if ( $author_html && $date_html ) {
        $meta_html = sprintf(
            '<div class="frp-meta">%s<span class="frp-meta-separator"> - </span>%s</div>',
            $author_html,
            $date_html
        );
    } elseif ( $author_html ) {
        $meta_html = sprintf( '<div class="frp-meta">%s</div>', $author_html );
    } elseif ( $date_html ) {
        $meta_html = sprintf( '<div class="frp-meta">%s</div>', $date_html );
    }

    // Kutipan (Excerpt)
    $excerpt_html = '';
    if ( $show_excerpt ) {
        $excerpt_text = get_the_excerpt( $post_id );
        
        if ( $excerpt_text ) {
            $final_excerpt_content = '';
            
            if ( mb_strlen( $excerpt_text ) > $limit_excerpt ) {
                $final_excerpt_content = mb_substr( $excerpt_text, 0, $limit_excerpt ) . '...';
            } else {
                $final_excerpt_content = $excerpt_text;
            }
            
            $final_excerpt_content = esc_html( $final_excerpt_content );

            if ( $show_more ) {
                $read_more_text = __( 'Read more &raquo;', 'frp' );
                $final_excerpt_content .= sprintf(
                    ' <a href="%s" class="frp-read-more">%s</a>',
                    esc_url( $post_link ),
                    esc_html( $read_more_text )
                );
            }

            $excerpt_html = sprintf( '<div class="frp-excerpt">%s</div>', $final_excerpt_content );
        }
    }

    // Gambar Unggulan
    $image_html = '';
    $image_style = '';
    $has_image = $show_image && has_post_thumbnail( $post_id );
    
    if ( $has_image ) {
        // Abaikan 'imagesize' jika mode adalah 3
        if ( $image_size_px > 0 && $mode !== 3 ) {
            $image_style = sprintf(
                'style="flex-basis: %dpx; width: %dpx; min-width: %dpx;"',
                $image_size_px,
                $image_size_px,
                $image_size_px
            );
        }
        
        $image_size_to_use = ( $mode === 3 ) ? 'medium' : 'thumbnail';
        
        $image_html = get_the_post_thumbnail( $post_id, $image_size_to_use );
    }

    // --- Membangun HTML berdasarkan mode ---
    
    $item_classes = sprintf( 'frp-item frp-mode-%d', $mode );
    
    if ( ! $has_image ) {
        $item_classes .= ' frp-no-image';
    }

    $content_part = sprintf(
        '<div class="frp-content-inner">%s%s%s%s</div>',
        $terms_html,
        $title_html,
        $meta_html,
        $excerpt_html
    );

    echo '<div class="' . esc_attr( $item_classes ) . '">';

    if ( ! $has_image ) {
        echo '<div class="frp-content-col">' . $content_part . '</div>';
    } else {
        switch ( $mode ) {
            case 1: // Gambar kiri
                printf(
                    '<div class="frp-image-col" %s><div class="frp-image-wrap">%s</div></div>',
                    $image_style,
                    $image_html
                );
                echo '<div class="frp-content-col">' . $content_part . '</div>';
                break;

            case 2: // Gambar kanan
                echo '<div class="frp-content-col">' . $content_part . '</div>';
                printf(
                    '<div class="frp-image-col" %s><div class="frp-image-wrap">%s</div></div>',
                    $image_style,
                    $image_html
                );
                break;
            
            case 3: // Gambar atas
            default:
                // 'imagesize' diabaikan, style dikontrol oleh CSS
                echo '<div class="frp-image-col"><div class="frp-image-wrap">' . $image_html . '</div></div>';
                echo '<div class="frp-content-col">' . $content_part . '</div>';
                break;
        }
    }

    echo '</div>';
}