<?php
/**
 * Plugin Name:       Flexi Recent Comments
 * Plugin URI:        https://mwahyunz.id/plugin
 * Description:       Display the latest comments anywhere using the shortcode [flexicomments]. Includes attributes for customization: avatar, number of comments, comment author, comment character limit, display post title, and comments date.
 * Version:           1.5.2.3
 * Author:            Mhd Wahyu NZ
 * Author URI:        https://mwahyunz.id
 * License:           GPLv2 or later
 * License URI:       http://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       frc
 * Domain Path:       /lang
 */

defined('ABSPATH') or die('No script kiddies please!');

/**
 * Main plugin class dengan Singleton pattern untuk performa optimal.
 */
final class Flexi_Recent_Comments {
    private static $instance = null;
    private static $cache_key_prefix = 'frc_comments_';
    private static $author_cache = null;
    private static $comment_cache = [];
    
    const CACHE_GROUP = 'frc';
    const SHORTCODE_CACHE_TIME = 300; // 5 minutes
    const QUERY_CACHE_TIME = 600; // 10 minutes
    const AUTHOR_CACHE_TIME = 3600; // 1 hour

    /**
     * Singleton instance getter.
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Private constructor - daftarkan hooks dan shortcode.
     */
    private function __construct() {
        add_action('plugins_loaded', [$this, 'load_textdomain']);
        add_action('wp_enqueue_scripts', [$this, 'register_assets']);
        add_shortcode('flexicomments', [$this, 'render_shortcode']);
        
        // Cache invalidation hooks
        add_action('comment_post', [$this, 'clear_cache']);
        add_action('edit_comment', [$this, 'clear_cache']);
        add_action('delete_comment', [$this, 'clear_cache']);
        add_action('transition_comment_status', [$this, 'clear_cache']);
        add_action('wp_set_comment_status', [$this, 'clear_cache']);
    }

    /**
     * Load textdomain untuk internasionalisasi.
     */
    public function load_textdomain() {
        load_plugin_textdomain('frc', false, dirname(plugin_basename(__FILE__)) . '/lang/');
    }

    /**
     * Register stylesheet dengan versioning otomatis.
     */
    public function register_assets() {
        wp_register_style(
            'flexi-recent-comments-style',
            plugins_url('flexi-recent-comments-style.css', __FILE__),
            [],
            '1.5.2.3'
        );
    }

    /**
     * Clear all plugin caches.
     */
    public function clear_cache() {
        wp_cache_flush_group(self::CACHE_GROUP);
        delete_transient('frc_author_ids');
        self::$author_cache = null;
    }

    /**
     * Render shortcode dengan caching dan sanitasi lengkap.
     */
    public function render_shortcode($atts) {
        wp_enqueue_style('flexi-recent-comments-style');

        // Parse dan validasi atribut dengan batasan maksimum
        $atts = shortcode_atts([
            'number' => 5,
            'avatar' => 1,
            'size'   => 50,
            'author' => 1,
            'limit'  => 80,
            'title'  => 1,
            'date'   => 1,
        ], $atts, 'flexicomments');

        $number = min(max(absint($atts['number']), 1), 50);
        $avatar = (bool) absint($atts['avatar']);
        $size   = min(max(absint($atts['size']), 20), 200);
        $author = (bool) absint($atts['author']);
        $limit  = min(max(absint($atts['limit']), 10), 500);
        $title  = (bool) absint($atts['title']);
        $date   = (bool) absint($atts['date']);

        // Cache berdasarkan kombinasi atribut
        $cache_key = self::$cache_key_prefix . md5(serialize([
            'number' => $number,
            'avatar' => $avatar,
            'size'   => $size,
            'author' => $author,
            'limit'  => $limit,
            'title'  => $title,
            'date'   => $date,
        ]));
        
        $cached_output = wp_cache_get($cache_key, self::CACHE_GROUP);

        if (false !== $cached_output) {
            return $cached_output;
        }

        // Ambil komentar dengan caching
        $comments = $this->get_cached_comments($number, $author);

        if (empty($comments)) {
            $output = '<p class="flexicomments_empty">' . esc_html__('No comments found.', 'frc') . '</p>';
            wp_cache_set($cache_key, $output, self::CACHE_GROUP, self::SHORTCODE_CACHE_TIME);
            return $output;
        }

        // Pre-load posts untuk menghindari N+1 queries
        $post_ids = array_unique(wp_list_pluck($comments, 'comment_post_ID'));
        $posts = $this->get_posts_batch($post_ids);

        // Render output dengan output buffering untuk performa
        ob_start();
        echo '<ul class="flexicomments_list">';
        foreach ($comments as $comment) {
            $post = isset($posts[$comment->comment_post_ID]) ? $posts[$comment->comment_post_ID] : null;
            echo $this->render_comment_item($comment, $post, $avatar, $size, $limit, $title, $date);
        }
        echo '</ul>';
        $output = ob_get_clean();

        // Cache selama 5 menit
        wp_cache_set($cache_key, $output, self::CACHE_GROUP, self::SHORTCODE_CACHE_TIME);

        return $output;
    }

    /**
     * Batch load posts untuk menghindari N+1 queries.
     * 
     * @param array $post_ids Array of post IDs
     * @return array Associative array of posts indexed by post ID
     */
    private function get_posts_batch($post_ids) {
        if (empty($post_ids)) {
            return [];
        }

        $cache_key = self::$cache_key_prefix . 'posts_' . md5(serialize($post_ids));
        $cached = wp_cache_get($cache_key, self::CACHE_GROUP);

        if (false !== $cached) {
            return $cached;
        }

        global $wpdb;
        $placeholders = implode(',', array_fill(0, count($post_ids), '%d'));
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $query = $wpdb->prepare(
            "SELECT ID, post_title, post_status FROM {$wpdb->posts} WHERE ID IN ($placeholders) AND post_status = 'publish'",
            $post_ids
        );
        
        $results = $wpdb->get_results($query);
        
        $posts = [];
        foreach ($results as $post) {
            $posts[(int)$post->ID] = $post;
        }

        wp_cache_set($cache_key, $posts, self::CACHE_GROUP, self::QUERY_CACHE_TIME);

        return $posts;
    }

    /**
     * Ambil komentar dengan object caching WordPress.
     */
    private function get_cached_comments($number, $include_author) {
        $cache_key = self::$cache_key_prefix . 'query_' . $number . '_' . (int)$include_author;
        $cached = wp_cache_get($cache_key, self::CACHE_GROUP);

        if (false !== $cached) {
            return $cached;
        }

        $args = [
            'status'        => 'approve',
            'number'        => $number,
            'type'          => 'comment',
            'orderby'       => 'comment_date_gmt',
            'order'         => 'DESC',
            'no_found_rows' => true,
            'update_comment_meta_cache' => false,
            'update_comment_post_cache' => false,
        ];

        if (!$include_author) {
            $excluded_ids = $this->get_excluded_author_ids();
            if (!empty($excluded_ids)) {
                $args['author__not_in'] = $excluded_ids;
            }
        }

        $comments = get_comments($args);
        wp_cache_set($cache_key, $comments, self::CACHE_GROUP, self::QUERY_CACHE_TIME);

        return $comments;
    }

    /**
     * Ambil author IDs yang di-exclude dengan transient caching.
     */
    private function get_excluded_author_ids() {
        if (null !== self::$author_cache) {
            return self::$author_cache;
        }

        $transient_key = 'frc_author_ids';
        $author_ids = get_transient($transient_key);

        if (false === $author_ids) {
            $author_ids = get_users([
                'capability__in' => ['publish_posts'],
                'fields'         => 'ID',
                'number'         => 100,
            ]);
            
            // Ensure we have an array of integers
            $author_ids = array_map('intval', $author_ids);
            
            set_transient($transient_key, $author_ids, self::AUTHOR_CACHE_TIME);
        }

        self::$author_cache = $author_ids;
        return $author_ids;
    }

    /**
     * Render single comment item dengan sanitasi dan escaping lengkap.
     * 
     * @param WP_Comment $comment Comment object
     * @param WP_Post|null $post Post object or null
     * @param bool $avatar Show avatar
     * @param int $size Avatar size
     * @param int $limit Comment character limit
     * @param bool $title Show post title
     * @param bool $date Show comment date
     * @return string HTML output
     */
    private function render_comment_item($comment, $post, $avatar, $size, $limit, $title, $date) {
        if (!$post || 'publish' !== $post->post_status) {
            return '';
        }

        $output = '<li class="flexicomments_item">';

        // Avatar - works for both registered users and guests
        if ($avatar) {
            $output .= '<div class="flexicomments_avatar_wrapper">';
            $avatar_html = get_avatar($comment, $size, '', '', [
                'class' => 'flexicomments_avatar',
                'loading' => 'lazy',
            ]);
            $output .= wp_kses_post($avatar_html);
            $output .= '</div>';
        }
        
        $output .= '<div class="flexicomments_content_wrapper">';

        // Comment author with safe link
        $comment_author = get_comment_author($comment);
        $comment_link = get_comment_link($comment);
        
        // Validate comment link
        if (!$comment_link || !filter_var($comment_link, FILTER_VALIDATE_URL)) {
            $comment_link = get_permalink($post);
        }

        $output .= sprintf(
            '<div class="flexicomments_author"><a href="%s" rel="nofollow noopener">%s</a></div>',
            esc_url($comment_link),
            esc_html($comment_author)
        );
        
        if ($title || $date) {
            $output .= '<div class="flexicomments_meta">';
            $meta_parts = [];

            if ($date) {
                $comment_date = get_comment_date(get_option('date_format'), $comment);
                $meta_parts[] = '<span class="flexicomments_date">' . esc_html($comment_date) . '</span>';
            }
            
            if ($title) {
                $post_title = get_the_title($post);
                if (empty($post_title)) {
                    /* translators: %d: Post ID */
                    $post_title = sprintf(__('Post #%d', 'frc'), $post->ID);
                }
                $post_link = get_permalink($post);
                $meta_parts[] = sprintf(
                    '<span class="flexicomments_post_title"><a href="%s">%s</a></span>',
                    esc_url($post_link),
                    esc_html($post_title)
                );
            }
            
            $output .= implode(' <span class="flexicomments_separator">•</span> ', $meta_parts);
            $output .= '</div>';
        }

        // Sanitasi konten komentar dengan proper encoding
        $comment_content = strip_tags($comment->comment_content);
        $comment_content = trim($comment_content);
        
        if (mb_strlen($comment_content, 'UTF-8') > $limit) {
            $comment_content = mb_substr($comment_content, 0, $limit, 'UTF-8');
            // Remove incomplete words at the end
            $comment_content = preg_replace('/\s+\S*$/', '', $comment_content);
            $comment_content .= '…'; // Proper UTF-8 ellipsis
        }

        $output .= '<div class="flexicomments_content">' . esc_html($comment_content) . '</div>';

        $output .= '</div>'; // close .flexicomments_content_wrapper
        $output .= '</li>';

        return $output;
    }

    /**
     * Prevent cloning of the instance.
     * 
     * @return void
     */
    public function __clone() {
        _doing_it_wrong(
            __FUNCTION__,
            esc_html__('Cloning instances of this class is forbidden.', 'frc'),
            '1.5.2.3'
        );
    }
    
    /**
     * Prevent unserializing of the instance.
     * 
     * @return void
     */
    public function __wakeup() {
        _doing_it_wrong(
            __FUNCTION__,
            esc_html__('Unserializing instances of this class is forbidden.', 'frc'),
            '1.5.2.3'
        );
    }
}

// Initialize plugin
Flexi_Recent_Comments::get_instance();