<?php
/**
 * Plugin Name:       Pluz Optimize Featured Image
 * Plugin URI:        https://mwahyunz.id/plugin/pluz-optimize-featured-image/
 * Description:       Speed up the loading of featured images on single WordPress posts.
 * Version:           1.0.9
 * Requires at least: 5.8
 * Requires PHP:      7.4
 * Author:            Mhd Wahyu NZ
 * Author URI:        https://mwahyunz.id
 * License:           GPL v2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       pofi
 * Domain Path:       /lang
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

final class Pluz_Optimize_Featured_Image_Final {
    private static $instance = null;
    private static $image_cache = [];
    private static $optimization_applied = false;

    const CACHE_GROUP = 'pofi';
    const CACHE_TIME = 3600; // 1 hour

    /**
     * Singleton instance.
     *
     * @return self
     */
    public static function get_instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Private constructor untuk Singleton pattern.
     */
    private function __construct() {
        // Delay initialization sampai WordPress siap
        add_action( 'template_redirect', [ $this, 'maybe_initialize_optimization' ], 1 );
    }

    /**
     * Cek kondisi dan mulai optimasi jika sesuai.
     *
     * @return void
     */
    public function maybe_initialize_optimization() {
        // Skip jika bukan kondisi yang tepat
        if ( is_admin() 
            || ! is_singular() 
            || ! has_post_thumbnail() 
            || self::$optimization_applied 
        ) {
            return;
        }

        // Cache data gambar sekali saja
        if ( ! $this->cache_featured_image_data() ) {
            return; // Tidak ada featured image valid
        }

        self::$optimization_applied = true;

        add_action( 'wp_head', [ $this, 'add_preload_featured_image' ], 1 );
        add_action( 'template_redirect', [ $this, 'start_output_buffering' ], 2 );
    }

    /**
     * Start output buffering untuk modifikasi HTML.
     *
     * @return void
     */
    public function start_output_buffering() {
        ob_start( [ $this, 'modify_final_html' ] );
    }

    /**
     * Cache semua data gambar untuk menghindari query berulang.
     *
     * @return bool True jika berhasil cache, false jika tidak ada image valid
     */
    private function cache_featured_image_data() {
        $post_id = get_queried_object_id();
        if ( ! $post_id ) {
            return false;
        }

        $image_id = get_post_thumbnail_id( $post_id );

        if ( ! $image_id || ! wp_attachment_is_image( $image_id ) ) {
            return false;
        }

        // Gunakan object cache WordPress
        $cache_key = 'image_data_' . $image_id;
        $cached_data = wp_cache_get( $cache_key, self::CACHE_GROUP );

        if ( false !== $cached_data && is_array( $cached_data ) ) {
            self::$image_cache = $cached_data;
            return true;
        }

        // Get image metadata
        $image_meta = wp_get_attachment_metadata( $image_id );
        if ( ! $image_meta || empty( $image_meta['file'] ) ) {
            return false;
        }

        $image_url = wp_get_attachment_image_url( $image_id, 'full' );
        if ( ! $image_url ) {
            return false;
        }

        self::$image_cache = [
            'image_id'       => $image_id,
            'image_url'      => $image_url,
            'image_srcset'   => wp_get_attachment_image_srcset( $image_id, 'full' ),
            'image_sizes'    => wp_get_attachment_image_sizes( $image_id, 'full' ),
            'image_basename' => basename( $image_meta['file'] ),
        ];

        // Cache selama 1 jam
        wp_cache_set( $cache_key, self::$image_cache, self::CACHE_GROUP, self::CACHE_TIME );

        return true;
    }

    /**
     * Tambahkan tag preload di <head> dengan escaping aman.
     *
     * @return void
     */
    public function add_preload_featured_image() {
        if ( empty( self::$image_cache['image_url'] ) ) {
            return;
        }

        $atts = [
            'href'          => esc_url( self::$image_cache['image_url'] ),
            'as'            => 'image',
            'fetchpriority' => 'high',
        ];

        if ( ! empty( self::$image_cache['image_srcset'] ) ) {
            $atts['imagesrcset'] = esc_attr( self::$image_cache['image_srcset'] );
        }

        if ( ! empty( self::$image_cache['image_sizes'] ) ) {
            $atts['imagesizes'] = esc_attr( self::$image_cache['image_sizes'] );
        }

        // Build preload tag
        $attr_strings = [];
        foreach ( $atts as $key => $value ) {
            $attr_strings[] = sprintf( '%s="%s"', esc_attr( $key ), $value );
        }

        printf(
            '<link rel="preload" %s />' . "\n",
            implode( ' ', $attr_strings )
        );
    }

    /**
     * Modifikasi HTML final dengan validasi dan error handling.
     *
     * @param string $buffer HTML buffer
     * @return string Modified HTML buffer
     */
    public function modify_final_html( $buffer ) {
        // Validasi buffer
        if ( empty( $buffer ) || ! is_string( $buffer ) ) {
            return $buffer;
        }

        if ( empty( self::$image_cache['image_basename'] ) ) {
            return $buffer;
        }

        // Cek apakah buffer adalah HTML valid
        if ( false === stripos( $buffer, '<html' ) && false === stripos( $buffer, '<!DOCTYPE' ) ) {
            return $buffer;
        }

        return $this->process_html_with_dom( $buffer );
    }

    /**
     * Process HTML menggunakan DOMDocument dengan safety checks.
     *
     * @param string $buffer HTML buffer
     * @return string Processed HTML buffer
     */
    private function process_html_with_dom( $buffer ) {
        // Konfigurasi DOMDocument untuk keamanan
        $dom = new DOMDocument();
        
        // Setup error handling
        $use_errors = libxml_use_internal_errors( true );
        
        // Security: Prevent XXE attacks
        $prev_entity_loader = libxml_disable_entity_loader( true );
        $dom->resolveExternals = false;
        $dom->substituteEntities = false;

        // Convert encoding untuk DOMDocument
        $html = mb_convert_encoding( $buffer, 'HTML-ENTITIES', 'UTF-8' );

        // Load HTML dengan flags optimal
        $loaded = $dom->loadHTML(
            $html,
            LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD | LIBXML_NOWARNING | LIBXML_NOERROR
        );

        // Restore error handling
        libxml_disable_entity_loader( $prev_entity_loader );
        
        if ( ! $loaded ) {
            libxml_clear_errors();
            libxml_use_internal_errors( $use_errors );
            return $buffer; // Fallback ke buffer asli jika gagal
        }

        // Process images
        $modified = $this->optimize_featured_image_in_dom( $dom );

        // Clear errors
        libxml_clear_errors();
        libxml_use_internal_errors( $use_errors );

        if ( ! $modified ) {
            return $buffer;
        }

        // Save HTML dengan error handling
        $output = $dom->saveHTML();
        
        return $output ?: $buffer;
    }

    /**
     * Optimize featured image dalam DOMDocument.
     *
     * @param DOMDocument $dom DOMDocument object
     * @return bool True jika ada modifikasi, false jika tidak
     */
    private function optimize_featured_image_in_dom( $dom ) {
        $images = $dom->getElementsByTagName( 'img' );
        
        if ( $images->length === 0 ) {
            return false;
        }

        $basename = self::$image_cache['image_basename'];
        
        // Buat pattern yang aman untuk matching
        $pattern = '/' . preg_quote( $basename, '/' ) . '(?![a-zA-Z0-9_-])/i';

        $found = false;

        foreach ( $images as $image ) {
            $src = $image->getAttribute( 'src' );
            $srcset = $image->getAttribute( 'srcset' );

            // Cek apakah ini featured image
            if ( ! preg_match( $pattern, $src ) && ! preg_match( $pattern, $srcset ) ) {
                continue;
            }

            // Apply optimizations
            $this->apply_optimization_attributes( $image );
            
            $found = true;
            break; // Hanya optimize satu featured image
        }

        return $found;
    }

    /**
     * Apply optimization attributes pada image element.
     *
     * @param DOMElement $image Image element
     * @return void
     */
    private function apply_optimization_attributes( $image ) {
        // Set priority attributes
        $image->setAttribute( 'fetchpriority', 'high' );
        $image->setAttribute( 'loading', 'eager' );
        
        // Remove conflicting attributes
        $image->removeAttribute( 'decoding' );
        $image->removeAttribute( 'data-src' );
        $image->removeAttribute( 'data-srcset' );
        $image->removeAttribute( 'data-lazy-src' );
        $image->removeAttribute( 'data-lazy-srcset' );

        // Clean up classes
        $this->clean_lazy_classes( $image );
    }

    /**
     * Clean lazy-loading classes dari image element.
     *
     * @param DOMElement $image Image element
     * @return void
     */
    private function clean_lazy_classes( $image ) {
        $class = $image->getAttribute( 'class' );
        
        if ( empty( $class ) ) {
            return;
        }

        // Parse classes
        $classes = array_filter( array_map( 'trim', explode( ' ', $class ) ) );
        
        // Remove lazy-loading classes
        $lazy_classes = [ 
            'lazy', 
            'lazyload', 
            'lazy-loaded', 
            'lazyloading',
            'lazyloaded',
            'lazy-load',
        ];
        
        $classes = array_diff( $classes, $lazy_classes );
        
        // Add no-lazy class
        $classes[] = 'no-lazy';
        $classes = array_unique( $classes );

        // Set cleaned classes
        if ( ! empty( $classes ) ) {
            $image->setAttribute( 'class', implode( ' ', $classes ) );
        }
    }

    /**
     * Prevent cloning of singleton instance.
     *
     * @return void
     */
    public function __clone() {
        _doing_it_wrong(
            __FUNCTION__,
            esc_html__( 'Cloning instances of this class is forbidden.', 'pofi' ),
            '1.0.9'
        );
    }

    /**
     * Prevent unserializing of singleton instance.
     *
     * @return void
     */
    public function __wakeup() {
        _doing_it_wrong(
            __FUNCTION__,
            esc_html__( 'Unserializing instances of this class is forbidden.', 'pofi' ),
            '1.0.9'
        );
    }
}

// Initialize plugin
Pluz_Optimize_Featured_Image_Final::get_instance();